/*
 * dumpmidi.c
 *
 *  Created on: Dec 31, 2012
 *      Author: David J. Rager
 *       Email: djrager@fourthwoods.com
 *
 * This code is hereby released into the public domain per the Creative Commons
 * Public Domain dedication.
 *
 * http://http://creativecommons.org/publicdomain/zero/1.0/
 */
#include <windows.h>
#include <mmsystem.h>
#include <stdio.h>

#define MAX_BUFFER_SIZE (512 * 12)
HANDLE event;

#pragma pack(push, 1)

struct _mid_header {
	unsigned int	id;		// identifier "MThd"
	unsigned int	size;	// always 6 in big-endian format
	unsigned short	format;	// big-endian format
	unsigned short  tracks;	// number of tracks, big-endian
	unsigned short	ticks;	// number of ticks per quarter note, big-endian
};

struct _mid_track {
	unsigned int	id;		// identifier "MTrk"
	unsigned int	length;	// track length, big-endian
};

#pragma pack(pop)

struct evt {
	unsigned int absolute_time;
	unsigned char* data;
	unsigned char event;
};

struct trk {
	struct _mid_track* track;
	unsigned char* buf;
	struct evt last_event;
	unsigned int absolute_time;
};

struct _mid_header* hdr = NULL;
HMIDIOUT out;
unsigned int PPQN_CLOCK;

static unsigned char* load_file(const unsigned char* filename, unsigned int* len)
{
	unsigned char* buf;
	unsigned int ret;
	FILE* f = fopen((char*)filename, "rb");
	if(f == NULL)
		return 0;

	fseek(f, 0, SEEK_END);
	*len = ftell(f);
	fseek(f, 0, SEEK_SET);

	buf = (unsigned char*)malloc(*len);
	if(buf == 0)
	{
		fclose(f);
		return 0;
	}

	ret = fread(buf, 1, *len, f);
	fclose(f);

	if(ret != *len)
	{
		free(buf);
		return 0;
	}

	return buf;
}

static unsigned long read_var_long(unsigned char* buf, unsigned int* bytesread)
{
	unsigned long var = 0;
	unsigned char c;

	*bytesread = 0;

	do
	{
		c = buf[(*bytesread)++];
		var = (var << 7) + (c & 0x7f);
	}
	while(c & 0x80);

	return var;
}

static unsigned short swap_bytes_short(unsigned short in)
{
	return ((in << 8) | (in >> 8));
}

static unsigned long swap_bytes_long(unsigned long in)
{
	unsigned short *p;
	p = (unsigned short*)&in;

	return (  (((unsigned long)swap_bytes_short(p[0])) << 16) |
				(unsigned long)swap_bytes_short(p[1]));
}

static struct evt get_next_event(const struct trk* track, int p)
{
	unsigned char* buf;
	struct evt e;
	unsigned int bytesread;
	unsigned int time;

	buf = track->buf;

	time = read_var_long(buf, &bytesread);
	buf += bytesread;

	e.absolute_time = track->absolute_time + time;
	e.data = buf;
	e.event = *e.data;

	return e;
}

static int is_track_end(const struct evt* e)
{
	if(e->event == 0xff) // meta-event?
		if(*(e->data + 1) == 0x2f) // track end?
			return 1;

	return 0;
}

extern unsigned char* notes[];
extern unsigned char* patch[];
extern unsigned char* percussion[];

static void print_event(struct evt evt)
{
	unsigned char event = evt.event & 0xf0;
	unsigned char channel = evt.event & 0x0f;
	unsigned char data1 = evt.data[1]; // get the first data byte
	unsigned char data2 = '\0';

	if(!(event == 0xc0 || (event == 0xd0)))
	{
		data2 = evt.data[2]; // get the second data byte
	}

	switch(event)
	{
	case 0x80:
		printf("Note off, channel %d, ", channel);
		printf("%s, %d\n", (channel != 9) ? notes[data1] : percussion[data1], data2);
		break;
	case 0x90:
		printf("Note on, channel %d, ", channel);
		printf("%s, %d\n", (channel != 9) ? notes[data1] : percussion[data1], data2);
		break;
	case 0xa0:
		printf("Key pressure, channel %d, ", channel);
		printf("%s, %d\n", (channel != 9) ? notes[data1] : percussion[data1], data2);
		break;
	case 0xb0:
		printf("Controller change, channel %d, ", channel);
		printf("%d, %d\n", data1, data2);
		break;
	case 0xc0:
		printf("Patch change, channel %d, ", channel);
		printf("%s\n", patch[data1]);
		break;
	case 0xd0:
		printf("Channel pressure, channel %d, ", channel);
		printf("%d\n", data1);
		break;
	case 0xe0:
		printf("Pitch wheel, channel %d, ", channel);
		printf("%04x\n", (data1 & 0x7f) | ((unsigned int)(data2 & 0x7f)) << 7);
		break;
	case 0xf0:
		printf("Meta-event\n");
		break;
	}
}

static void usleep(int waitTime);

static unsigned int print_tracks(struct trk* tracks, unsigned int ntracks)
{
	unsigned int i;
	static unsigned int current_time = 0;

	if(tracks == NULL)
		return 0;

	for(i = 0; i < ntracks; i++)
	{
		printf("Track %d, length: %d\n", i, swap_bytes_long(tracks[i].track->length));
	}

	while(TRUE)
	{
		unsigned int time = (unsigned int)-1;
		unsigned int idx = -1;
		unsigned int bytesread = 0;

		struct evt evt;
		unsigned char c;

		unsigned int msg;
		unsigned int err;

		bytesread = 0;

		// get the next event
		for(i = 0; i < ntracks; i++)
		{
		//	if(i != 1)
		//		continue;
			evt = get_next_event(&tracks[i], 0);
			if(!(is_track_end(&evt)) && (evt.absolute_time <= time))
			{
				time = evt.absolute_time;
				idx = i;
			}
		}

		// if idx == -1 then all the tracks have been read up to the end of track mark
		if(idx == -1)
			break; // we're done

		evt = get_next_event(&tracks[idx], (idx == 2) ? 1 : 0);

		tracks[idx].absolute_time = evt.absolute_time;
		time = tracks[idx].absolute_time - current_time;
		current_time = tracks[idx].absolute_time;

		printf("Track %d: ticks: %d: ", idx, tracks[idx].absolute_time);
		
		usleep(time * PPQN_CLOCK);

		if(!(evt.event & 0x80)) // running mode
		{
			struct evt last = tracks[idx].last_event;
			last.data = (unsigned char*)malloc(4);
			last.data[0] = last.event;
			last.data[1] = evt.data[0];
			last.data[2] = evt.data[1];

			printf("+ ");

			print_event(last);
			free(last.data);
			
			c = evt.data[bytesread++]; // get the first data byte

			msg = 0 | ((unsigned long)MEVT_SHORTMSG << 24) |
						((unsigned long)last.event) |
						((unsigned long)c << 8);
			if(!((last.event & 0xf0) == 0xc0 || (last.event & 0xf0) == 0xd0))
			{
				c = evt.data[bytesread++]; // get the second data byte
				msg |= ((unsigned long)c << 16);
			}

	  		err = midiOutShortMsg(out, msg);
 
			tracks[idx].buf = evt.data + bytesread;
		}
		else if(evt.event == 0xff) // meta-event
		{
			print_event(evt);

			bytesread++; // skip the event byte
			unsigned char meta = evt.data[bytesread++]; // read the meta-event byte
			unsigned int len;

			switch(meta)
			{
			case 0x51: // only care about tempo events
				{
					unsigned char a, b, c;
					len = evt.data[bytesread++]; // get the length byte, should be 3
					a = evt.data[bytesread++];
					b = evt.data[bytesread++];
					c = evt.data[bytesread++];

					msg = 0 | ((unsigned long)a << 16) |
						((unsigned long)b << 8) |
						((unsigned long)c << 0);

					PPQN_CLOCK = msg / swap_bytes_short(hdr->ticks);
				}
				break;
			default: // skip all other meta events
				len = evt.data[bytesread++]; // get the length byte
				bytesread += len;
				break;
			}

			tracks[idx].buf = evt.data + bytesread;
		}
		else if((evt.event & 0xf0) != 0xf0) // normal command
		{
			print_event(evt);
			tracks[idx].last_event = evt;
			bytesread++; // skip the event byte
			c = evt.data[bytesread++]; // get the first data byte

			msg = 0 | ((unsigned long)MEVT_SHORTMSG << 24) |
				((unsigned long)evt.event << 0) |
				((unsigned long)c << 8);

			if(!((evt.event & 0xf0) == 0xc0 || (evt.event & 0xf0) == 0xd0))
			{
				c = evt.data[bytesread++]; // get the second data byte
				msg |= ((unsigned long)c << 16);
			}

			err = midiOutShortMsg(out, msg);

			tracks[idx].buf = evt.data + bytesread;
		}

	}

	return 1;
}

static void usleep(int waitTime) {
	LARGE_INTEGER time1, time2, freq;

	if(waitTime == 0)
		return;

	QueryPerformanceCounter(&time1);
	QueryPerformanceFrequency(&freq);

	do {
		QueryPerformanceCounter(&time2);
	} while((time2.QuadPart - time1.QuadPart) * 1000000ll / freq.QuadPart < waitTime);
}

int main(int argc, char* argv[])
{
	unsigned char* midibuf = NULL;
	unsigned int midilen = 0;

	unsigned int err, msg;
	unsigned int PPQN_CLOCK;
	unsigned int i;

	unsigned short ntracks = 0;
	struct trk* tracks = NULL;

	if(argc <= 1)
	{
		printf("no\n");
		return 0;
	}

	err = midiOutOpen(&out, 0, 0, 0, CALLBACK_NULL);
	if (err != MMSYSERR_NOERROR)
		printf("error opening default MIDI device: %d\n", err);
	else
		printf("successfully opened default MIDI device\n");

	midibuf = load_file((unsigned char*)argv[1], &midilen);
	if(midibuf == NULL)
	{
		printf("could not open %s\n", argv[1]);
		return 0;
	}

	hdr = (struct _mid_header*)midibuf;
	midibuf += sizeof(struct _mid_header);
	ntracks = swap_bytes_short(hdr->tracks);

	PPQN_CLOCK = 500000 / swap_bytes_short(hdr->ticks);

	printf("Format: %d\n", swap_bytes_short(hdr->format));
	printf("Tracks: %d\n", swap_bytes_short(hdr->tracks));
	printf("Ticks: %d\n", swap_bytes_short(hdr->ticks));

	tracks = (struct trk*)malloc(ntracks * sizeof(struct trk));
	if(tracks == NULL)
		goto error1;

	for(i = 0; i < ntracks; i++)
	{
		tracks[i].track = (struct _mid_track*)midibuf;
		tracks[i].buf = midibuf + sizeof(struct _mid_track);
		tracks[i].absolute_time = 0;
		tracks[i].last_event;

		midibuf += sizeof(struct _mid_track) + swap_bytes_long(tracks[i].track->length);
	}

	print_tracks(tracks, ntracks);

error2:
	free(tracks);

error1:
	free(hdr);

	return(0);
}

unsigned char* notes[] = {
	"C - 0", "C# - 0", "D - 0", "D# - 0", "E - 0", "F - 0", "F# - 0", "G - 0", "G# - 0", "A - 0", "A# - 0", "B - 0",
	"C - 1", "C# - 1", "D - 1", "D# - 1", "E - 1", "F - 1", "F# - 1", "G - 1", "G# - 1", "A - 1", "A# - 1", "B - 1",
	"C - 2", "C# - 2", "D - 2", "D# - 2", "E - 2", "F - 2", "F# - 2", "G - 2", "G# - 2", "A - 2", "A# - 2", "B - 2",
	"C - 3", "C# - 3", "D - 3", "D# - 3", "E - 3", "F - 3", "F# - 3", "G - 3", "G# - 3", "A - 3", "A# - 3", "B - 3",
	"C - 4", "C# - 4", "D - 4", "D# - 4", "E - 4", "F - 4", "F# - 4", "G - 4", "G# - 4", "A - 4", "A# - 4", "B - 4",
	"C - 5", "C# - 5", "D - 5", "D# - 5", "E - 5", "F - 5", "F# - 5", "G - 5", "G# - 5", "A - 5", "A# - 5", "B - 5",
	"C - 6", "C# - 6", "D - 6", "D# - 6", "E - 6", "F - 6", "F# - 6", "G - 6", "G# - 6", "A - 6", "A# - 6", "B - 6",
	"C - 7", "C# - 7", "D - 7", "D# - 7", "E - 7", "F - 7", "F# - 7", "G - 7", "G# - 7", "A - 7", "A# - 7", "B - 7",
	"C - 8", "C# - 8", "D - 8", "D# - 8", "E - 8", "F - 8", "F# - 8", "G - 8", "G# - 8", "A - 8", "A# - 8", "B - 8",
	"C - 9", "C# - 9", "D - 9", "D# - 9", "E - 9", "F - 9", "F# - 9", "G - 9", "G# - 9", "A - 9", "A# - 9", "B - 9",
	"C - 10", "C# - 10", "D - 10", "D# - 10", "E - 10", "F - 10", "F# - 10", "G - 10"
};

unsigned char* patch[] = {
"Acoustic Grand",
"Bright Acoustic",
"Electric Grand",
"Honky-Tonk",
"Electric Piano 1",
"Electric Piano 2",
"Harpsichord",
"Clav",
"Celesta",
"Glockenspiel",
"Music Box",
"Vibraphone",
"Marimba",
"Xylophone",
"Tubular Bells",
"Dulcimer",
"Drawbar Organ",
"Percussive Organ",
"Rock Organ",
"Church Organ",
"Reed Organ",
"Accoridan",
"Harmonica",
"Tango Accordian",
"Acoustic Guitar(nylon)",
"Acoustic Guitar(steel)",
"Electric Guitar(jazz)",
"Electric Guitar(clean)",
"Electric Guitar(muted)",
"Overdriven Guitar",
"Distortion Guitar",
"Guitar Harmonics",
"Acoustic Bass",
"Electric Bass(finger)",
"Electric Bass(pick)",
"Fretless Bass",
"Slap Bass 1",
"Slap Bass 2",
"Synth Bass 1",
"Synth Bass 2",
"Violin",
"Viola",
"Cello",
"Contrabass",
"Tremolo Strings",
"Pizzicato Strings",
"Orchestral Strings",
"Timpani",
"String Ensemble 1",
"String Ensemble 2",
"SynthStrings 1",
"SynthStrings 2",
"Choir Aahs",
"Voice Oohs",
"Synth Voice",
"Orchestra Hit",
"Trumpet",
"Trombone",
"Tuba",
"Muted Trumpet",
"French Horn",
"Brass Section",
"SynthBrass 1",
"SynthBrass 2",
"Soprano Sax",
"Alto Sax",
"Tenor Sax",
"Baritone Sax",
"Oboe",
"English Horn",
"Bassoon",
"Clarinet",
"Piccolo",
"Flute",
"Recorder",
"Pan Flute",
"Blown Bottle",
"Skakuhachi",
"Whistle",
"Ocarina",
"Lead 1 (square)",
"Lead 2 (sawtooth)",
"Lead 3 (calliope)",
"Lead 4 (chiff)",
"Lead 5 (charang)",
"Lead 6 (voice)",
"Lead 7 (fifths)",
"Lead 8 (bass+lead)",
"Pad 1 (new age)",
"Pad 2 (warm)",
"Pad 3 (polysynth)",
"Pad 4 (choir)",
"Pad 5 (bowed)",
"Pad 6 (metallic)",
"Pad 7 (halo)",
"Pad 8 (sweep)",
"FX 1 (rain)",
"FX 2 (soundtrack)",
"FX 3 (crystal)",
"FX 4 (atmosphere)",
"FX 5 (brightness)",
"FX 6 (goblins)",
"FX 7 (echoes)",
"FX 8 (sci-fi)",
"Sitar",
"Banjo",
"Shamisen",
"Koto",
"Kalimba",
"Bagpipe",
"Fiddle",
"Shanai",
"Tinkle Bell",
"Agogo",
"Steel Drums",
"Woodblock",
"Taiko Drum",
"Melodic Tom",
"Synth Drum",
"Reverse Cymbal",
"Guitar Fret Noise",
"Breath Noise",
"Seashore",
"Bird Tweet",
"Telephone Ring",
"Helicopter",
"Applause",
"Gunshot" 
};

unsigned char* percussion[] = {
"", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
"Acoustic Bass Drum",
"Bass Drum 1",
"Side Stick",
"Acoustic Snare",
"Hand Clap",
"Electric Snare",
"Low Floor Tom",
"Closed Hi-Hat",
"High Floor Tom",
"Pedal Hi-Hat",
"Low Tom",
"Open Hi-Hat",
"Low-Mid Tom",
"Hi-Mid Tom",
"Crash Cymbal 1",
"High Tom",
"Ride Cymbal 1",
"Chinese Cymbal",
"Ride Bell",
"Tambourine",
"Splash Cymbal",
"Cowbell",
"Crash Cymbal 2",
"Vibraslap",
"Ride Cymbal 2",
"Hi Bongo",
"Low Bongo",
"Mute Hi Conga",
"Open Hi Conga",
"Low Conga",
"High Timbale",
"Low Timbale",
"High Agogo",
"Low Agogo",
"Cabasa",
"Maracas",
"Short Whistle",
"Long Whistle",
"Short Guiro",
"Long Guiro",
"Claves",
"Hi Wood Block",
"Low Wood Block",
"Mute Cuica",
"Open Cuica",
"Mute Triangle",
"Open Triangle"
};
